/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.data;

import ghidra.framework.data.AbstractTransactionManager;
import ghidra.framework.data.DomainObjectAdapterDB;
import ghidra.framework.data.DomainObjectDBTransaction;
import ghidra.framework.model.AbortedTransactionListener;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.TransactionInfo;
import ghidra.framework.model.TransactionListener;
import ghidra.util.Msg;
import ghidra.util.Swing;
import ghidra.util.SystemUtilities;
import ghidra.util.datastruct.WeakDataStructureFactory;
import ghidra.util.datastruct.WeakSet;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

class DomainObjectTransactionManager
extends AbstractTransactionManager {
    private LinkedList<DomainObjectDBTransaction> undoList = new LinkedList();
    private LinkedList<DomainObjectDBTransaction> redoList = new LinkedList();
    private WeakSet<TransactionListener> transactionListeners = WeakDataStructureFactory.createCopyOnWriteWeakSet();
    private DomainObjectAdapterDB domainObj;
    private DomainObjectAdapterDB[] domainObjAsArray;
    private DomainObjectDBTransaction transaction;

    DomainObjectTransactionManager(DomainObjectAdapterDB domainObj) {
        this.domainObj = domainObj;
        domainObj.dbh.setMaxUndos(50);
        this.domainObjAsArray = new DomainObjectAdapterDB[]{domainObj};
    }

    DomainObjectAdapterDB getDomainObject() {
        return this.domainObj;
    }

    @Override
    DomainObjectAdapterDB[] getDomainObjects() {
        return this.domainObjAsArray;
    }

    @Override
    void checkDomainObject(DomainObjectAdapterDB object) {
        if (object != this.domainObj) {
            throw new IllegalArgumentException("invalid domain object");
        }
    }

    @Override
    void clearTransactions() {
        this.domainObj.dbh.setMaxUndos(0);
        this.domainObj.dbh.setMaxUndos(50);
        if (this.domainObj.changeSet != null) {
            this.domainObj.changeSet.clearUndo();
        }
        this.undoList.clear();
        this.redoList.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    void terminateTransaction(boolean rollback, boolean notify) {
        DomainObjectTransactionManager domainObjectTransactionManager = this;
        synchronized (domainObjectTransactionManager) {
            if (this.transaction == null || this.transactionTerminated) {
                return;
            }
            try {
                this.domainObj.dbh.terminateTransaction(this.transaction.getID(), !rollback);
            }
            catch (IOException e) {
                this.domainObj.dbError(e);
            }
            this.transaction.abort();
            this.transactionTerminated = true;
            if (this.domainObj.changeSet != null) {
                this.domainObj.changeSet.endTransaction(!rollback);
            }
            this.domainObj.domainObjectRestored();
            if (notify) {
                this.notifyEndTransaction();
            }
        }
    }

    @Override
    synchronized int startTransaction(DomainObjectAdapterDB object, String description, AbortedTransactionListener listener, boolean force, boolean notify) {
        if (object != this.domainObj) {
            throw new IllegalArgumentException("invalid domain object");
        }
        if (!force) {
            this.verifyNoLock();
        }
        if (this.transaction == null) {
            this.transactionTerminated = false;
            this.transaction = new DomainObjectDBTransaction(this.domainObj.dbh.startTransaction(), this.domainObj);
            if (this.domainObj.changeSet != null) {
                this.domainObj.changeSet.startTransaction();
            }
            int id = this.transaction.addEntry(description, listener);
            if (notify) {
                this.notifyStartTransaction(this.transaction);
            }
            return id;
        }
        if (this.transactionTerminated) {
            Msg.warn((Object)this, (Object)("Aborted transaction still pending, new transaction will also be aborted: " + description));
        }
        return this.transaction.addEntry(description, listener);
    }

    private void flushDomainObjectEvents() {
        SystemUtilities.runSwingLater(() -> this.domainObj.flushEvents());
    }

    @Override
    synchronized TransactionInfo endTransaction(DomainObjectAdapterDB object, int transactionID, boolean commit, boolean notify) throws IllegalStateException {
        if (object != this.domainObj) {
            throw new IllegalArgumentException("invalid domain object");
        }
        if (this.transaction == null) {
            throw new IllegalStateException("No transaction is open");
        }
        DomainObjectDBTransaction returnedTransaction = this.transaction;
        try {
            this.transaction.endEntry(transactionID, commit && !this.transactionTerminated);
            TransactionInfo.Status status = this.transaction.getStatus();
            if (status == TransactionInfo.Status.COMMITTED) {
                object.flushWriteCache();
                boolean committed = this.domainObj.dbh.endTransaction(this.transaction.getID(), true);
                if (committed) {
                    returnedTransaction.setHasCommittedDBTransaction();
                    this.domainObj.changed = true;
                    this.redoList.clear();
                    this.undoList.addLast(this.transaction);
                    if (this.undoList.size() > 50) {
                        this.undoList.removeFirst();
                    }
                    this.flushDomainObjectEvents();
                }
                if (this.domainObj.changeSet != null) {
                    this.domainObj.changeSet.endTransaction(committed);
                }
                if (notify) {
                    this.notifyEndTransaction();
                }
                this.transaction = null;
            } else if (status == TransactionInfo.Status.ABORTED) {
                object.invalidateWriteCache();
                if (!this.transactionTerminated) {
                    this.domainObj.dbh.endTransaction(this.transaction.getID(), false);
                    if (this.domainObj.changeSet != null) {
                        this.domainObj.changeSet.endTransaction(false);
                    }
                }
                this.domainObj.domainObjectRestored();
                this.transaction.restoreToolStates(true);
                this.transaction = null;
                if (notify) {
                    this.notifyEndTransaction();
                }
            }
        }
        catch (IOException e) {
            this.transaction = null;
            this.domainObj.dbError(e);
        }
        return returnedTransaction;
    }

    @Override
    int getUndoStackDepth() {
        return this.undoList.size();
    }

    @Override
    synchronized boolean canRedo() {
        if (this.transaction == null && this.redoList.size() > 0) {
            return this.domainObj.dbh.canRedo();
        }
        return false;
    }

    @Override
    synchronized boolean canUndo() {
        if (this.transaction == null && this.undoList.size() > 0) {
            return this.domainObj.dbh.canUndo();
        }
        return false;
    }

    @Override
    synchronized String getRedoName() {
        if (this.transaction == null && this.redoList.size() > 0) {
            TransactionInfo t = this.redoList.getLast();
            return t.getDescription();
        }
        return "";
    }

    @Override
    synchronized String getUndoName() {
        if (this.transaction == null && this.undoList.size() > 0) {
            TransactionInfo t = this.undoList.getLast();
            return t.getDescription();
        }
        return "";
    }

    @Override
    List<String> getAllUndoNames() {
        return this.getDescriptions(this.undoList);
    }

    @Override
    List<String> getAllRedoNames() {
        return this.getDescriptions(this.redoList);
    }

    private List<String> getDescriptions(List<DomainObjectDBTransaction> list) {
        ArrayList<String> descriptions = new ArrayList<String>();
        for (DomainObjectDBTransaction tx : list) {
            descriptions.add(tx.getDescription());
        }
        Collections.reverse(descriptions);
        return descriptions;
    }

    @Override
    TransactionInfo getCurrentTransactionInfo() {
        return this.transaction;
    }

    @Override
    void doRedo(boolean notify) throws IOException {
        if (this.canRedo()) {
            DomainObjectDBTransaction t = this.redoList.removeLast();
            this.domainObj.dbh.redo();
            this.domainObj.clearCache(false);
            if (this.domainObj.changeSet != null) {
                this.domainObj.changeSet.redo();
            }
            this.undoList.addLast(t);
            this.domainObj.domainObjectRestored();
            t.restoreToolStates(false);
            if (notify) {
                this.notifyUndoRedo();
            }
        }
    }

    @Override
    void doUndo(boolean notify) throws IOException {
        if (this.canUndo()) {
            DomainObjectDBTransaction t = this.undoList.removeLast();
            t.initAfterState(this.domainObj);
            this.domainObj.dbh.undo();
            if (this.domainObj.changeSet != null) {
                this.domainObj.changeSet.undo();
            }
            this.redoList.addLast(t);
            this.domainObj.domainObjectRestored();
            t.restoreToolStates(true);
            if (notify) {
                this.notifyUndoRedo();
            }
        }
    }

    @Override
    synchronized void clearUndo(boolean notifyListeners) {
        if (!this.undoList.isEmpty() || !this.redoList.isEmpty()) {
            this.undoList.clear();
            this.redoList.clear();
            DomainFile df = this.domainObj.getDomainFile();
            if (this.domainObj.changeSet != null) {
                this.domainObj.changeSet.clearUndo(df != null && df.isCheckedOut());
            }
            if (notifyListeners) {
                this.notifyUndoStackChanged();
            }
        }
    }

    @Override
    void doClose(DomainObjectAdapterDB object) {
    }

    @Override
    void addTransactionListener(DomainObjectAdapterDB object, TransactionListener listener) {
        if (object != this.domainObj) {
            throw new IllegalArgumentException("invalid domain object");
        }
        this.transactionListeners.add((Object)listener);
    }

    @Override
    void removeTransactionListener(DomainObjectAdapterDB object, TransactionListener listener) {
        if (object != this.domainObj) {
            throw new IllegalArgumentException("invalid domain object");
        }
        this.transactionListeners.remove((Object)listener);
    }

    void notifyStartTransaction(TransactionInfo tx) {
        Swing.runLater(() -> {
            for (TransactionListener listener : this.transactionListeners) {
                listener.transactionStarted(this.domainObj, tx);
                listener.undoStackChanged(this.domainObj);
            }
        });
    }

    void notifyEndTransaction() {
        Swing.runLater(() -> {
            for (TransactionListener listener : this.transactionListeners) {
                listener.transactionEnded(this.domainObj);
                listener.undoStackChanged(this.domainObj);
            }
        });
    }

    void notifyUndoStackChanged() {
        Swing.runLater(() -> {
            for (TransactionListener listener : this.transactionListeners) {
                listener.undoStackChanged(this.domainObj);
            }
        });
    }

    void notifyUndoRedo() {
        Swing.runLater(() -> {
            for (TransactionListener listener : this.transactionListeners) {
                listener.undoRedoOccurred(this.domainObj);
                listener.undoStackChanged(this.domainObj);
            }
        });
    }
}

