# Copyright (C) 2000-2004 Stichting LogReport Foundation LogReport@LogReport.org

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program (see COPYING); if not, check with
# http://www.gnu.org/copyleft/gpl.html.
#
# Authors:
#   Egon Willighagen <egonw@logreport.org>
#   Wessel Dankers <wsl@logreport.org>
#   Francis J. Lacoste <flacoste@logreport.org>

package Lire::DlfConverters::CUPS;

use strict;

use base qw/Lire::DlfConverter/;
use Lire::Time qw/clf2cal/;
use Carp;

sub new {
    return bless {}, shift;
}

sub name {
    return 'cups_pagelog';
}

sub title {
    return 'CUPS printer server page log';
}

sub description { 
    return '<para>This DlfConverter converts CUPS page
log file to the print schema.</para>';
}

sub schemas {
    return qw/print/;
}

sub handle_log_lines {
    return 1;
}

sub init_dlf_converter {
    my ($self, $process) = @_;
    $self->{'dlf'} = {};
    return;
}

sub process_log_line {
    my ($self, $process, $line) = @_;

    # snoopy ANGELA 2 [09/Jun/2005:12:50:37 +0200] 1 1 - 10.27.1.223
    # lp joostvb 87 [22/Jul/2006:13:37:27 +0200] 1 1 - localhost
    my ( $printer, $user, $job_id, $clftime, $pagenum, $ncopies, $billing )
      = $line =~ /^([^ ]*) ([^ ]*) ([^ ]*) (\[[^]]+\]) ([^ ]*) ([^ ]*)(?: -)?(?:\s([^ ]*))?$/;
    unless ( $printer ) {
        $process->error( 'invalid page_log line', $line );
        return;
    }

    my $time = eval { clf2cal( $clftime ) };
    if ( $@ ) {
        $process->error( $@, $line );
        return;
    }

    if( !defined $self->{'dlf'}{'job-id'}
        || $self->{'dlf'}{'job-id'} ne $job_id )
    {
        # new job; output the DLF line
        $process->write_dlf('print', $self->{'dlf'})
          if defined $self->{'dlf'}{'job-id'};

        $self->{'dlf'} = { 'printer' => $printer,
                           'time' => $time,
                           'duration' => 0,
                           'num_pages' => $pagenum,
                           'num_copies' => $ncopies,
                           'job-id' => $job_id,
                           'billing' => undef,
                           'user' => undef,
                         };
        $self->{'dlf'}{'billing'} = $billing if $billing;
        $self->{'dlf'}{'user'} = $user if $user;
    } else {
        # same job; update info
        $self->{'dlf'}{'num_pages'} = $pagenum;
        $self->{'dlf'}{'duration'} = $time - $self->{'dlf'}{'time'};
    }

    return;
}

sub finish_conversion {
    my ($self, $process) = @_;

    $process->write_dlf('print', $self->{'dlf'})
      if defined $self->{'dlf'}{'job-id'};

    return;
}

1;
